/*
 * Copyright (c) 2004 bartavelle
 * bartavelle@bandecon.com
 *
 * Simple MD5 hashes cracker
 * It uses the Solar Designer's md5 implementation
 *
 * Minor changes by David Luyer <david@luyer.net> to
 * use a modified (faster) version of Solar Designer's
 * md5 implementation.
 *
 * More improvement by
 * Balzs Bucsay - earthquake@rycon.hu - http://www.rycon.hu/
 * (2times faster, but it's only works up to 54characters)
 *
 */

#include <string.h>

#include "arch.h"
#include "misc.h"
#include "common.h"
#include "formats.h"

#if !ARCH_LITTLE_ENDIAN
#define MD5_out MD5_bitswapped_out
#endif


#define FORMAT_LABEL			"raw-md5_szakdoga"
#define FORMAT_NAME			"Raw MD5 Szakdolgozat"
#define ALGORITHM_NAME			"raw-md5_szakdoga"

#define BENCHMARK_COMMENT		""
#define BENCHMARK_LENGTH		-1

#define PLAINTEXT_LENGTH		32
#define CIPHERTEXT_LENGTH		32

#define BINARY_SIZE			16
#define SALT_SIZE			0

#define MIN_KEYS_PER_CRYPT		1
#define MAX_KEYS_PER_CRYPT		64

extern ARCH_WORD_32 MD5_out[MAX_KEYS_PER_CRYPT];
extern char MD5_tmp[MAX_KEYS_PER_CRYPT][CIPHERTEXT_LENGTH + 1];

typedef unsigned int MD5_u32plus;

extern void MD5_Go_szakdoga(unsigned char *data, unsigned int len, int index);
extern void MD5_Go2_szakdoga(unsigned char *data, unsigned int len, int index);

static struct fmt_tests rawmd5_szakdoga_tests[] =
	{
		{"5a105e8b9d40e1329780d62ea2265d8a", "test1"},
		{"ad0234829205b9033196ba818f7a872b", "test2"},
		{"8ad8757baa8564dc136c1e07507f4a98", "test3"},
		{"86985e105f79b95d6bc918fb45ec7727", "test4"},
		{"378e2c4a07968da2eca692320136433d", "thatsworking"},
		{NULL}
	};

static char saved_key[MAX_KEYS_PER_CRYPT][PLAINTEXT_LENGTH + 1 + 128 /* MD5 scratch space */];
unsigned int saved_key_len[MAX_KEYS_PER_CRYPT];

static int valid(char *ciphertext)
{
	unsigned int i;

	if (strlen(ciphertext) != CIPHERTEXT_LENGTH)
		return 0;

	for (i = 0; i < CIPHERTEXT_LENGTH; i++) {
		if (!(  (('0' <= ciphertext[i])&&(ciphertext[i] <= '9')) ||
						(('a' <= ciphertext[i])&&(ciphertext[i] <= 'f'))  ))
			return 0;
	}

	return 1;
}

static void rawmd5_szakdoga_set_salt(void *salt)
{

}

static void rawmd5_szakdoga_set_key(char *key, int index)
{
	strnzcpy(saved_key[index], key, PLAINTEXT_LENGTH+1);
	saved_key_len[index] = strlen(saved_key[index]);
}

static char *rawmd5_szakdoga_get_key(int index)
{
	saved_key[index][saved_key_len[index]] = 0;
	return saved_key[index];
}

static int rawmd5_szakdoga_cmp_one(void *binary, int index)
{
	return (!(*((unsigned int*)binary) - (unsigned int)MD5_out[index]));
}

static int rawmd5_szakdoga_cmp_all(void *binary, int count)
{
	unsigned int i;

	for (i = 0; i < count; i++) {
		if (!(*((unsigned int*)binary) - *((unsigned int*)&MD5_out[i])))
			return 1;
	}

	return 0;
}

static int rawmd5_szakdoga_cmp_exact(char *source, int index)
{
	MD5_Go2_szakdoga((unsigned char *)saved_key[index], saved_key_len[index], index);
	return !memcmp(source, MD5_tmp[index], CIPHERTEXT_LENGTH);
}

static void rawmd5_szakdoga_crypt_all(int count)
{
	unsigned int i;

	for (i = 0; i < count; i++) {
		MD5_Go_szakdoga((unsigned char *)saved_key[i], saved_key_len[i], i);
	}
}

int rawmd5_szakdoga_binary_hash_0(void *binary)
{
	return *(ARCH_WORD_32 *)binary & 0xF;
}

int rawmd5_szakdoga_binary_hash_1(void *binary)
{
	return *(ARCH_WORD_32 *)binary & 0xFF;
}

int rawmd5_szakdoga_binary_hash_2(void *binary)
{
	return *(ARCH_WORD_32 *)binary & 0xFFF;
}

int rawmd5_szakdoga_get_hash_0(int index)
{
	return MD5_out[index] & 0xF;
}

int rawmd5_szakdoga_get_hash_1(int index)
{
	return MD5_out[index] & 0xFF;
}

int rawmd5_szakdoga_get_hash_2(int index)
{
	return MD5_out[index] & 0xFFF;
}

static void *rawmd5_szakdoga_binary(char *ciphertext)
{
	static char realcipher[BINARY_SIZE];
	unsigned int i;

	for (i=0;i<BINARY_SIZE;i++) {
		realcipher[i] = atoi16[ARCH_INDEX(ciphertext[i*2])]*16 + atoi16[ARCH_INDEX(ciphertext[i*2+1])];
	}

	return (void *)realcipher;
}

struct fmt_main fmt_rawMD5_szakdoga =
	{
		{
			FORMAT_LABEL,
			FORMAT_NAME,
			ALGORITHM_NAME,
			BENCHMARK_COMMENT,
			BENCHMARK_LENGTH,
			PLAINTEXT_LENGTH,
			BINARY_SIZE,
			SALT_SIZE,
			MIN_KEYS_PER_CRYPT,
			MAX_KEYS_PER_CRYPT,
			FMT_CASE | FMT_8_BIT,
			rawmd5_szakdoga_tests
		}

		, {
			fmt_default_init,
			valid,
			fmt_default_split,
			rawmd5_szakdoga_binary,
			fmt_default_salt,
			{
				rawmd5_szakdoga_binary_hash_0,
				rawmd5_szakdoga_binary_hash_1,
				rawmd5_szakdoga_binary_hash_2
			},
			fmt_default_salt_hash,
			rawmd5_szakdoga_set_salt,
			rawmd5_szakdoga_set_key,
			rawmd5_szakdoga_get_key,
			fmt_default_clear_keys,
			rawmd5_szakdoga_crypt_all,
			{
				rawmd5_szakdoga_get_hash_0,
				rawmd5_szakdoga_get_hash_1,
				rawmd5_szakdoga_get_hash_2
			},
			rawmd5_szakdoga_cmp_all,
			rawmd5_szakdoga_cmp_one,
			rawmd5_szakdoga_cmp_exact
		}
	};
