/*
 * This is an OpenSSL-compatible implementation of the RSA Data Security,
 * Inc. MD5 Message-Digest Algorithm.
 *
 * Written by Solar Designer <solar@openwall.com> in 2001, and placed in
 * the public domain.  There's absolutely no warranty.
 *
 * This differs from Colin Plumb's older public domain implementation in
 * that no 32-bit integer data type is required, there's no compile-time
 * endianness configuration, and the function prototypes match OpenSSL's.
 * The primary goals are portability and ease of use.
 *
 * This implementation is meant to be fast, but not as fast as possible.
 * Some known optimizations are not included to reduce source code size
 * and avoid compile-time configuration.
 *
 * This file has been modifed by David Luyer <david@luyer.net> to introduce
 * some performance improvements, at the cost of its general-purpose use.
 * See the caveats documented above the MD5_Go() routine.
 *
 * New performance improvements by Balzs Bucsay. It's only works up to 54
 * characters, but it is just enough.
 * http://www.rycon.hu/ - earthquake@rycon.hu
 *
 */

#include <string.h>

#include "arch.h"
#include "misc.h"
#include "common.h"

#define BINARY_SIZE			16
#define CIPHERTEXT_LENGTH		32
#define MAX_KEYS_PER_CRYPT		64

/* Output words */

ARCH_WORD_32 MD5_out[MAX_KEYS_PER_CRYPT];
char MD5_tmp[MAX_KEYS_PER_CRYPT][CIPHERTEXT_LENGTH + 1];
ARCH_WORD_32 MD5_bitswapped_out2[4];
#if !ARCH_LITTLE_ENDIAN
ARCH_WORD_32 MD5_bitswapped_out[MAX_KEYS_PER_CRYPT];
#endif

/* Bit-swapped output words */

#if !defined(_MD5_GO_H)
#define _MD5_GO_H

/* Any 32-bit or wider unsigned integer data type will do */
typedef unsigned int MD5_u32plus;

#endif


/*
 * The basic MD5 functions.
 *
 * F is optimized compared to its RFC 1321 definition just like in Colin
 * Plumb's implementation.
 */
#define F(x, y, z)			((z) ^ ((x) & ((y) ^ (z))))
#define G(x, y, z)			((y) ^ ((z) & ((x) ^ (y))))
#define H(x, y, z)			((x) ^ (y) ^ (z))
#define I(x, y, z)			((y) ^ ((x) | ~(z)))

/*
 * The MD5 transformation for all four rounds.
 */
#define STEP(f, a, b, c, d, x, t, s) \
	(a) += f((b), (c), (d)) + (x) + (t); \
	(a) = (((a) << (s)) | (((a) & 0xffffffff) >> (32 - (s)))); \
	(a) += (b);

/*
 * SET reads 4 input bytes in little-endian byte order and stores them
 * in a properly aligned word in host byte order.
 *
 * The check for little-endian architectures which tolerate unaligned
 * memory accesses is just an optimization.  Nothing will break if it
 * doesn't work.
 */

#if ARCH_ALLOWS_UNALIGNED
#define SET(n) \
	(*(MD5_u32plus *)&ptr[(n) * 4])
#define GET(n) \
	SET(n)
#else

static MD5_u32plus work[16];

#define SET(n) \
	(work[(n)] = \
	(MD5_u32plus)ptr[(n) * 4] | \
	((MD5_u32plus)ptr[(n) * 4 + 1] << 8) | \
	((MD5_u32plus)ptr[(n) * 4 + 2] << 16) | \
	((MD5_u32plus)ptr[(n) * 4 + 3] << 24))
#define GET(n) \
	(work[(n)])
#endif

/*
 * This processes one or more 64-byte data blocks, but does NOT update
 * the bit counters.  There're no alignment requirements.
 */
static void body2_szakdoga(void *data, int index)
{
	unsigned char *ptr;
	unsigned int i;
	MD5_u32plus a, b, c, d;

	ptr = data;

	a = 0x67452301;
	b = 0xefcdab89;
	c = 0x98badcfe;
	d = 0x10325476;

	/* Round 1 */
	STEP(F, a, b, c, d, SET(0), 0xd76aa478, 7)
	STEP(F, d, a, b, c, SET(1), 0xe8c7b756, 12)
	STEP(F, c, d, a, b, SET(2), 0x242070db, 17)
	STEP(F, b, c, d, a, SET(3), 0xc1bdceee, 22)
	STEP(F, a, b, c, d, SET(4), 0xf57c0faf, 7)
	STEP(F, d, a, b, c, SET(5), 0x4787c62a, 12)
	STEP(F, c, d, a, b, SET(6), 0xa8304613, 17)
	STEP(F, b, c, d, a, SET(7), 0xfd469501, 22)
	STEP(F, a, b, c, d, SET(8), 0x698098d8, 7)
	STEP(F, d, a, b, c, SET(9), 0x8b44f7af, 12)
	STEP(F, c, d, a, b, SET(10), 0xffff5bb1, 17)
	STEP(F, b, c, d, a, SET(11), 0x895cd7be, 22)
	STEP(F, a, b, c, d, SET(12), 0x6b901122, 7)
	STEP(F, d, a, b, c, SET(13), 0xfd987193, 12)
	STEP(F, c, d, a, b, SET(14), 0xa679438e, 17)
	STEP(F, b, c, d, a, SET(15), 0x49b40821, 22)

	/* Round 2 */
	STEP(G, a, b, c, d, GET(1), 0xf61e2562, 5)
	STEP(G, d, a, b, c, GET(6), 0xc040b340, 9)
	STEP(G, c, d, a, b, GET(11), 0x265e5a51, 14)
	STEP(G, b, c, d, a, GET(0), 0xe9b6c7aa, 20)
	STEP(G, a, b, c, d, GET(5), 0xd62f105d, 5)
	STEP(G, d, a, b, c, GET(10), 0x02441453, 9)
	STEP(G, c, d, a, b, GET(15), 0xd8a1e681, 14)
	STEP(G, b, c, d, a, GET(4), 0xe7d3fbc8, 20)
	STEP(G, a, b, c, d, GET(9), 0x21e1cde6, 5)
	STEP(G, d, a, b, c, GET(14), 0xc33707d6, 9)
	STEP(G, c, d, a, b, GET(3), 0xf4d50d87, 14)
	STEP(G, b, c, d, a, GET(8), 0x455a14ed, 20)
	STEP(G, a, b, c, d, GET(13), 0xa9e3e905, 5)
	STEP(G, d, a, b, c, GET(2), 0xfcefa3f8, 9)
	STEP(G, c, d, a, b, GET(7), 0x676f02d9, 14)
	STEP(G, b, c, d, a, GET(12), 0x8d2a4c8a, 20)

	/* Round 3 */
	STEP(H, a, b, c, d, GET(5), 0xfffa3942, 4)
	STEP(H, d, a, b, c, GET(8), 0x8771f681, 11)
	STEP(H, c, d, a, b, GET(11), 0x6d9d6122, 16)
	STEP(H, b, c, d, a, GET(14), 0xfde5380c, 23)
	STEP(H, a, b, c, d, GET(1), 0xa4beea44, 4)
	STEP(H, d, a, b, c, GET(4), 0x4bdecfa9, 11)
	STEP(H, c, d, a, b, GET(7), 0xf6bb4b60, 16)
	STEP(H, b, c, d, a, GET(10), 0xbebfbc70, 23)
	STEP(H, a, b, c, d, GET(13), 0x289b7ec6, 4)
	STEP(H, d, a, b, c, GET(0), 0xeaa127fa, 11)
	STEP(H, c, d, a, b, GET(3), 0xd4ef3085, 16)
	STEP(H, b, c, d, a, GET(6), 0x04881d05, 23)
	STEP(H, a, b, c, d, GET(9), 0xd9d4d039, 4)
	STEP(H, d, a, b, c, GET(12), 0xe6db99e5, 11)
	STEP(H, c, d, a, b, GET(15), 0x1fa27cf8, 16)
	STEP(H, b, c, d, a, GET(2), 0xc4ac5665, 23)

	/* Round 4 */
	STEP(I, a, b, c, d, GET(0), 0xf4292244, 6)
	STEP(I, d, a, b, c, GET(7), 0x432aff97, 10)
	STEP(I, c, d, a, b, GET(14), 0xab9423a7, 15)
	STEP(I, b, c, d, a, GET(5), 0xfc93a039, 21)
	STEP(I, a, b, c, d, GET(12), 0x655b59c3, 6)
	STEP(I, d, a, b, c, GET(3), 0x8f0ccc92, 10)
	STEP(I, c, d, a, b, GET(10), 0xffeff47d, 15)
	STEP(I, b, c, d, a, GET(1), 0x85845dd1, 21)
	STEP(I, a, b, c, d, GET(8), 0x6fa87e4f, 6)
	STEP(I, d, a, b, c, GET(15), 0xfe2ce6e0, 10)
	STEP(I, c, d, a, b, GET(6), 0xa3014314, 15)
	STEP(I, b, c, d, a, GET(13), 0x4e0811a1, 21)
	STEP(I, a, b, c, d, GET(4), 0xf7537e82, 6)
	STEP(I, d, a, b, c, GET(11), 0xbd3af235, 10)
	STEP(I, c, d, a, b, GET(2), 0x2ad7d2bb, 15)
	STEP(I, b, c, d, a, GET(9), 0xeb86d391, 21)

	a += 0x67452301;
	b += 0xefcdab89;
	c += 0x98badcfe;
	d += 0x10325476;

#if ARCH_LITTLE_ENDIAN

	MD5_bitswapped_out2[0] = a;
	MD5_bitswapped_out2[1] = b;
	MD5_bitswapped_out2[2] = c;
	MD5_bitswapped_out2[3] = d;

#else

	MD5_bitswapped_out2[0] = (a << 24) |
													 (a >> 24) |
													 ((a << 8) & 0x00ff0000) |
													 ((a >> 8) & 0x0000ff00);

	MD5_bitswapped_out2[1] = (b << 24) |
													 (b >> 24) |
													 ((b << 8) & 0x00ff0000) |
													 ((b >> 8) & 0x0000ff00);

	MD5_bitswapped_out2[2] = (c << 24) |
													 (c >> 24) |
													 ((c << 8) & 0x00ff0000) |
													 ((c >> 8) & 0x0000ff00);

	MD5_bitswapped_out2[3] = (d << 24) |
													 (d >> 24) |
													 ((d << 8) & 0x00ff0000) |
													 ((d >> 8) & 0x0000ff00);

#endif

	for (i = 0; i < BINARY_SIZE; i++) {
		MD5_tmp[index][i * 2 + 0] = itoa16[ARCH_INDEX(((unsigned char*) MD5_bitswapped_out2)[i] / 16)];
		MD5_tmp[index][i * 2 + 1] = itoa16[ARCH_INDEX(((unsigned char*) MD5_bitswapped_out2)[i] & 0xf)];
	}
}

static void body_szakdoga(void *data, int index)
{
	unsigned char *ptr;
	MD5_u32plus a, b, c, d;

	ptr = data;

	a = 0x67452301;
	b = 0xefcdab89;
	c = 0x98badcfe;
	d = 0x10325476;

	/* Round 1 */
	STEP(F, a, b, c, d, SET(0), 0xd76aa478, 7)
	STEP(F, d, a, b, c, SET(1), 0xe8c7b756, 12)
	STEP(F, c, d, a, b, SET(2), 0x242070db, 17)
	STEP(F, b, c, d, a, SET(3), 0xc1bdceee, 22)
	STEP(F, a, b, c, d, SET(4), 0xf57c0faf, 7)
	STEP(F, d, a, b, c, SET(5), 0x4787c62a, 12)
	STEP(F, c, d, a, b, SET(6), 0xa8304613, 17)
	STEP(F, b, c, d, a, SET(7), 0xfd469501, 22)
	STEP(F, a, b, c, d, SET(8), 0x698098d8, 7)
	STEP(F, d, a, b, c, SET(9), 0x8b44f7af, 12)
	STEP(F, c, d, a, b, SET(10), 0xffff5bb1, 17)
	STEP(F, b, c, d, a, SET(11), 0x895cd7be, 22)
	STEP(F, a, b, c, d, SET(12), 0x6b901122, 7)
	STEP(F, d, a, b, c, SET(13), 0xfd987193, 12)
	STEP(F, c, d, a, b, SET(14), 0xa679438e, 17)
	STEP(F, b, c, d, a, SET(15), 0x49b40821, 22)

	/* Round 2 */
	STEP(G, a, b, c, d, GET(1), 0xf61e2562, 5)
	STEP(G, d, a, b, c, GET(6), 0xc040b340, 9)
	STEP(G, c, d, a, b, GET(11), 0x265e5a51, 14)
	STEP(G, b, c, d, a, GET(0), 0xe9b6c7aa, 20)
	STEP(G, a, b, c, d, GET(5), 0xd62f105d, 5)
	STEP(G, d, a, b, c, GET(10), 0x02441453, 9)
	STEP(G, c, d, a, b, GET(15), 0xd8a1e681, 14)
	STEP(G, b, c, d, a, GET(4), 0xe7d3fbc8, 20)
	STEP(G, a, b, c, d, GET(9), 0x21e1cde6, 5)
	STEP(G, d, a, b, c, GET(14), 0xc33707d6, 9)
	STEP(G, c, d, a, b, GET(3), 0xf4d50d87, 14)
	STEP(G, b, c, d, a, GET(8), 0x455a14ed, 20)
	STEP(G, a, b, c, d, GET(13), 0xa9e3e905, 5)
	STEP(G, d, a, b, c, GET(2), 0xfcefa3f8, 9)
	STEP(G, c, d, a, b, GET(7), 0x676f02d9, 14)
	STEP(G, b, c, d, a, GET(12), 0x8d2a4c8a, 20)

	/* Round 3 */
	STEP(H, a, b, c, d, GET(5), 0xfffa3942, 4)
	STEP(H, d, a, b, c, GET(8), 0x8771f681, 11)
	STEP(H, c, d, a, b, GET(11), 0x6d9d6122, 16)
	STEP(H, b, c, d, a, GET(14), 0xfde5380c, 23)
	STEP(H, a, b, c, d, GET(1), 0xa4beea44, 4)
	STEP(H, d, a, b, c, GET(4), 0x4bdecfa9, 11)
	STEP(H, c, d, a, b, GET(7), 0xf6bb4b60, 16)
	STEP(H, b, c, d, a, GET(10), 0xbebfbc70, 23)
	STEP(H, a, b, c, d, GET(13), 0x289b7ec6, 4)
	STEP(H, d, a, b, c, GET(0), 0xeaa127fa, 11)
	STEP(H, c, d, a, b, GET(3), 0xd4ef3085, 16)
	STEP(H, b, c, d, a, GET(6), 0x04881d05, 23)
	STEP(H, a, b, c, d, GET(9), 0xd9d4d039, 4)
	STEP(H, d, a, b, c, GET(12), 0xe6db99e5, 11)
	STEP(H, c, d, a, b, GET(15), 0x1fa27cf8, 16)
	STEP(H, b, c, d, a, GET(2), 0xc4ac5665, 23)

	/* Round 4 */
	STEP(I, a, b, c, d, GET(0), 0xf4292244, 6)
	STEP(I, d, a, b, c, GET(7), 0x432aff97, 10)
	STEP(I, c, d, a, b, GET(14), 0xab9423a7, 15)
	STEP(I, b, c, d, a, GET(5), 0xfc93a039, 21)
	STEP(I, a, b, c, d, GET(12), 0x655b59c3, 6)
	STEP(I, d, a, b, c, GET(3), 0x8f0ccc92, 10)
	STEP(I, c, d, a, b, GET(10), 0xffeff47d, 15)
	STEP(I, b, c, d, a, GET(1), 0x85845dd1, 21)
	STEP(I, a, b, c, d, GET(8), 0x6fa87e4f, 6)
	STEP(I, d, a, b, c, GET(15), 0xfe2ce6e0, 10)
	STEP(I, c, d, a, b, GET(6), 0xa3014314, 15)
	STEP(I, b, c, d, a, GET(13), 0x4e0811a1, 21)
	STEP(I, a, b, c, d, GET(4), 0xf7537e82, 6)

	MD5_out[index] = a + 0x67452301;
}

void MD5_Go_szakdoga(unsigned char *data, unsigned int len, int index)
{
	data[len] = 0x80;
	memset(&data[len+1], 0, 63 - len);
	data[56] = len << 3;
	data[57] = len >> 5;
	body_szakdoga(data, index);

#if !ARCH_LITTLE_ENDIAN
	MD5_bitswapped_out[index] = (MD5_out[index] << 24) |
															(MD5_out[index] >> 24) |
															((MD5_out[index] << 8) & 0x00ff0000) |
															((MD5_out[index] >> 8) & 0x0000ff00);
#endif
}

void MD5_Go2_szakdoga(unsigned char *data, unsigned int len, int index)
{
	data[len] = 0x80;
	memset(&data[len+1], 0, 63 - len);
	data[56] = len << 3;
	data[57] = len >> 5;
	body2_szakdoga(data, index);
}
