/* mysql_szakdoga_half_fmt.c
 *
 * Copyright (c) 2008 by <earthquake@rycon.hu>
 *
 * John the ripper MYSQL-fast module
 *
 *
 * Note: The mysql hash's first 8byte is relevant,
 * the another ones depends on the first 8. Maybe
 * the passwords after 9-10character have collision
 * in the first 8byte, so we have to check the full
 * hash.
 *
 * Unbelievable good optimization by Pter Kasza
 *
 * http://rycon.hu/
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "arch.h"
#include "misc.h"
#include "common.h"
#include "formats.h"

#define FORMAT_LABEL			"mysql-szakdoga"
#define FORMAT_NAME				"mysql_szakdoga"
#define ALGORITHM_NAME			"mysql-szakdoga"

#define BENCHMARK_COMMENT		""
#define BENCHMARK_LENGTH		-1

#define PLAINTEXT_LENGTH		32
#define CIPHERTEXT_LENGTH		16

#define BINARY_SIZE			8
#define SALT_SIZE			0

#define MIN_KEYS_PER_CRYPT		1
#define MAX_KEYS_PER_CRYPT		64

unsigned int sz_add[PLAINTEXT_LENGTH];
unsigned long sz_nr[PLAINTEXT_LENGTH];
char sz_password[PLAINTEXT_LENGTH+1];

static struct fmt_tests mysql_szakdoga_tests[] = {
	// ciphertext, plaintext
	{"445ff82636a7ba59", "probe"},
	{"60671c896665c3fa", "a"},
	{"1acbed4a27b20da3", "hash"},
	{"77ff75006118bab8", "hacker"},
	{"1b38cd9c2f809809", "hacktivity2008"},
	{"1b38cd9c2f809809", "hacktivity 2008"},
	{"6fc81597422015a8", "johnmodule"},
	{NULL}
};

static char crypt_key[MAX_KEYS_PER_CRYPT][BINARY_SIZE];
static char saved_key[MAX_KEYS_PER_CRYPT][PLAINTEXT_LENGTH + 1];

void mysql_szakdoga_init(void)
{
	memset(sz_password, 0, 17);
	memset(sz_add, 0, 16*sizeof(int));
	memset(sz_nr, 0, 16*sizeof(long int));
	
	sz_nr[0]=1345345333L;
	sz_add[0]=7;
}

static int mysql_szakdoga_valid(char* ciphertext)
{
	unsigned int i;

	if (strlen(ciphertext) != CIPHERTEXT_LENGTH)
		return 0;

	for (i = 0; i < CIPHERTEXT_LENGTH; i++)
	{
		if (!(((ciphertext[i] >= '0') && (ciphertext[i] <= '9')) ||
			((ciphertext[i] >= 'a') && (ciphertext[i] <= 'f'))))
			return 0;
	}

	return 1;
}

static void mysql_szakdoga_set_salt(void* salt) { }

static void* mysql_szakdoga_get_binary(char* ciphertext)
{
	static unsigned char buff[BINARY_SIZE / 2];
	unsigned int i;

	for (i = 0; i < BINARY_SIZE / 2; i++)
	{
#if ARCH_LITTLE_ENDIAN == 1
		buff[((BINARY_SIZE / 2) - 1) - i] = atoi16[ARCH_INDEX(ciphertext[i * 2])] * 16 + atoi16[ARCH_INDEX(ciphertext[i * 2 + 1])];
#else
		buff[i] = atoi16[ARCH_INDEX(ciphertext[i * 2])] * 16 + atoi16[ARCH_INDEX(ciphertext[i * 2 + 1])];
#endif
	}

	return buff;
}

static void mysql_szakdoga_set_key(char* key, int index)
{
	strnzcpy(saved_key[index], key, PLAINTEXT_LENGTH + 1);
}

static char* mysql_szakdoga_get_key(int index)
{
	return saved_key[index];
}

static int mysql_szakdoga_cmp_one(void* binary, int index)
{
	return (!(*((unsigned long*)binary) - *((unsigned long*)&crypt_key[index])));
}

static int mysql_szakdoga_cmp_all(void* binary, int count)
{
	unsigned int i;

	for (i = 0; i < count; i++)
	{
		if (!(*((unsigned long*)binary) - *((unsigned long*)&crypt_key[i])))
			return 1;
	}

	return 0;
}

static int mysql_szakdoga_cmp_exact(char* source, int index)
{
	register unsigned long nr = 1345345333L, add = 7, nr2 = 0x12345671L;
	register unsigned long tmp;
	char* password;
	char ctmp[CIPHERTEXT_LENGTH+1];

	password = saved_key[index];
	for (; *password; password++)
	{
		if (*password == ' ' || *password == '\t')
			continue;

		tmp = (unsigned long) (unsigned char) *password;
		nr ^= (((nr & 63) + add) * tmp) + (nr << 8);
		nr2 += (nr2 << 8) ^ nr;
		add += tmp;
	}

	sprintf(ctmp, "%08lx%08lx", (nr & (((unsigned long) 1L << 31) -1L)), (nr2 & (((unsigned long) 1L << 31) -1L)));
	return !memcmp(source, ctmp, CIPHERTEXT_LENGTH);
}

static void mysql_szakdoga_crypt_all(int count)
{
	unsigned long nr, add;
	unsigned long tmp;
	unsigned int i, j, len;
	char* password;

	for (i = 0; i < count; i++)
	{
		len = strlen(saved_key[i]);
		for (j=0;j<len-2;j++)
		{
		  if (saved_key[i][j] - sz_password[j]) 
		  {	  
			/*strncpy*/memcpy(sz_password+j, &saved_key[i][j], len-j+1);
			break;
		  }
		}
		nr=sz_nr[j];
		add=sz_add[j];

		password = saved_key[i]+j;
		for (; *password; password++)
		{
			if (*password == ' ' || *password == '\t')
				continue;

			tmp = (unsigned long) (unsigned char) *password;
			nr ^= (((nr & 63) + add) * tmp) + (nr << 8);
			add += tmp;
			j++;			
			sz_nr[j] = nr;
			sz_add[j] = add;
		}

		*((unsigned long*)&crypt_key[i]) = (nr & (((unsigned long) 1L << 31) -1L));
	}
}

int mysql_szakdoga_binary_hash_0(void *binary)
{
	return *(int *)binary & 0xF;
}

int mysql_szakdoga_binary_hash_1(void *binary)
{
	return *(int *)binary & 0xFF;
}

int mysql_szakdoga_binary_hash_2(void *binary)
{
	return *(int *)binary & 0xFFF;
}

int mysql_szakdoga_get_hash_0(int index)
{
	return *(int *)crypt_key[index] & 0xF;
}

int mysql_szakdoga_get_hash_1(int index)
{
	return *(int *)crypt_key[index] & 0xFF;
}

int mysql_szakdoga_get_hash_2(int index)
{
	return *(int *)crypt_key[index] & 0xFFF;
}

struct fmt_main fmt_MYSQL_szakdoga =
{
	{
		FORMAT_LABEL,
		FORMAT_NAME,
		ALGORITHM_NAME,
		BENCHMARK_COMMENT,
		BENCHMARK_LENGTH,
		PLAINTEXT_LENGTH,
		BINARY_SIZE,
		SALT_SIZE,
		MIN_KEYS_PER_CRYPT,
		MAX_KEYS_PER_CRYPT,
		FMT_CASE | FMT_8_BIT,
		mysql_szakdoga_tests
	}, {
		mysql_szakdoga_init,
		mysql_szakdoga_valid,
		fmt_default_split,
		mysql_szakdoga_get_binary,
		fmt_default_salt,
		{
			mysql_szakdoga_binary_hash_0,
			mysql_szakdoga_binary_hash_1,
			mysql_szakdoga_binary_hash_2
		},
		fmt_default_salt_hash,
		mysql_szakdoga_set_salt,
		mysql_szakdoga_set_key,
		mysql_szakdoga_get_key,
		fmt_default_clear_keys,
		mysql_szakdoga_crypt_all,
		{
			mysql_szakdoga_get_hash_0,
			mysql_szakdoga_get_hash_1,
			mysql_szakdoga_get_hash_2
		},
		mysql_szakdoga_cmp_all,
		mysql_szakdoga_cmp_one,
		mysql_szakdoga_cmp_exact
	}
};
